% endo_ms_lik_2nd_kim_nelson.m
% 
% gets posterior and filtered/smooth estimates for the model without
% stochastic volatility
% 
% Estimating Macroeconomic Models of Financial Crises: An Endogenous Regime-Switching Approach*
%   Gianluca Benigno, Andrew Foerster, Christopher Otrok, Alessandro Rebucci
% 
% Updated July 2024 
% % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % %
% 
% INPUTS
%   parameters  = a vector of parameters at which to evaluate likelihood
%   data_y      = (NY x T) matrix of data for estimating the likelihood
%   kf_filtering_level = scalar determining filter output, with 
%           if 0, outputs the updated variables
%           if 1, outputs the the smoothed variables
%   posterior   = scalar determining log-likelihood or posterior
%           if 0, outputs based on log-likelihood
%           if 1, outputs based on posterior (requires priorval function)
%   negative    = scalar determining log-likelihood output
%           if 0, outputs the log-likelihood
%           if 1, outputs the negative of the log-likelihood (for
%                       minimizations)
%   GLOBALS     = structure with many indices and dimensions    
%   
% 
% OUTPUTS
%   LLF         = scalar, value of the log-likelihood (or its negative)
%   Filters     = structure, elements depending on 'kf_filtering_level'
%       PAI     = (ns x T) matrix, predicted probabilities across regimes
%       Q       = (ns x ns x T) array, predicted transition matrix
%     	a       = (NX x T x ns x ns) array, predicted state variables
%       P       = (NX x NX x T x ns x ns) array, predicted covariance matrix
%       att     = (NX x T x ns) array, filtered state variables
%       Ptt     = (NX x NX x T x ns) array, filtered covariance matrix
%       PAItt   = (ns x T) matrix, filtered probabilities across regimes
%       atT     = (NX x T x ns) array, smoothed state variables
%       PAItT   = (ns x T) matrix, smoothed probabilities across regimes
%   Solution    = structure with solution to the model
%   retcode     = boolean, 1 solution found
% 
% % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % %
function [LLF_or_Posterior, Filters, Solution, retcode] ...
    = endo_ms_lik_2nd_kim_nelson(parameters,data_y,kf_filtering_level,posterior,negative,GLOBALS)

% -- System Dimensions -- %
ny = GLOBALS.ny; % # of control variables
nx = GLOBALS.nx; % # of state variables
ne = GLOBALS.ne; % # of structural shocks
ns = GLOBALS.ns; % # of regimes


% -- Estimation Dimensions -- %
NY = GLOBALS.estimation.NY; % observables in estimation
NX = GLOBALS.estimation.NX; % state dimension in estimation

% -- Setting Parameter Vector -- %
if posterior == 1
    [prior,rflag] = priorval(parameters,GLOBALS);
        
    if rflag == 1
        Filters     = 0;
        Solution    = 0;
        retcode     = 0;
        if negative == 1
            LLF_or_Posterior = Inf;
        else
            LLF_or_Posterior = -Inf;
        end        
        return;
    end
else
    prior = 1;
end

% -- Solve the Model -- %
try
    [xss,yss,sss,Pss,H,G,H2,G2,retcode] = endogenous_ms_solver_main(GLOBALS,parameters);
    Solution.xss    = xss;
    Solution.yss    = yss;
    Solution.sss    = sss;
    Solution.Pss    = Pss;
    Solution.H      = H;
    Solution.G      = G;
    Solution.H2     = H2;
    Solution.G2     = G2;
catch
    retcode = 0;
end

% -- If No Solution, Set Function Outputs and Exit -- %
if retcode == 0 
    Filters         = 0;
    Solution        = 0;  
    if negative == 1
        LLF_or_Posterior = Inf;
    else
        LLF_or_Posterior = -Inf;
    end
    return;
end

% -- Prior on 
meanstable  = switch_stochasticss(GLOBALS,xss,yss,H,G,H2,G2,1:nx+ny);
if GLOBALS.hierarchicalprior == 1
    ggamma00 = log(GLOBALS.parameters.p01/(1-GLOBALS.parameters.p01));
    ggamma01 = exp(parameters(GLOBALS.parm_ind.loggamma01));  
    bstar_stochss = meanstable(nx+GLOBALS.var_ind.y.bstar,1);
    
    p01_stochss = exp(ggamma00 - ggamma01*bstar_stochss)/(1 + exp(ggamma00 - ggamma01*bstar_stochss));
    [a, b]=betaparametertransform(0.01,0.025^2);
    hprior1 = betapdf(0.99999999999999999999999999999999999999999999999999999999^(-1)*p01_stochss,a,b);

    if(isinf(hprior1)); hprior1=0; end
    if(p01_stochss>0.1); hprior1=0; end
    
    prior = prior*hprior1;
    
end


    
% -- Observation Equation Selection Matrix-- %
D = GLOBALS.estimation.D;
E = GLOBALS.estimation.E;
                                        
% Specification for State Space System 
DD  = zeros(NY,NX,ns);
ED  = zeros(NY,NY,ns);
HD  = zeros(nx,nx+ne+1,ns);
GD  = zeros(ny,nx+ne+1,ns);
HD2 = zeros(nx,(nx+ne+1)^2,ns);
GD2 = zeros(ny,(nx+ne+1)^2,ns);
for ii = 1:ns
    DD(:,:,ii) = D;
    ED(:,:,ii) = E*E';
    HD(:,:,ii) = H{ii};
    GD(:,:,ii) = G{ii};
    HD2(:,:,ii) = H2{ii};
    GD2(:,:,ii) = G2{ii};
end

% Exogenous Shocks with Unit Variance by Assumption
V = eye(ne);

% Transition Matrix Generator (Can be Endogenous Function)
try
    ggammas(1,1)    = log(GLOBALS.parameters.p01/(1-GLOBALS.parameters.p01));
    ggammas(2,1)	= log(GLOBALS.parameters.p10/(1-GLOBALS.parameters.p10));
catch
    ggammas(1,1)    = log(parameters(GLOBALS.parm_ind.p01)/(1-parameters(GLOBALS.parm_ind.p01)));
    ggammas(2,1)	= log(parameters(GLOBALS.parm_ind.p10)/(1-parameters(GLOBALS.parm_ind.p10)));
end

try
    ggammas(1,2)    = exp(GLOBALS.parameters.loggamma01);
    ggammas(2,2)	= exp(GLOBALS.parameters.loggamma11);
catch      
    ggammas(1,2)    = exp(parameters(GLOBALS.parm_ind.loggamma01));    
    ggammas(2,2)	= exp(parameters(GLOBALS.parm_ind.loggamma11));
end


% Initial Conditions

a           = zeros(NX,2);
P(:,:,1)    = 1e-8*eye(NX);
P(:,:,2)    = 1e-8*eye(NX);

a(:,1) = [meanstable(1:nx,1);meanstable(1:nx,1);meanstable(nx+1:nx+ny,1);meanstable(nx+1:nx+ny,1);zeros(ne,1)];
a(:,2) = [meanstable(1:nx,2);meanstable(1:nx,2);meanstable(nx+1:nx+ny,2);meanstable(nx+1:nx+ny,2);zeros(ne,1)];
P(2*nx+2*ny+1:2*nx+2*ny+ne,2*nx+2*ny+1:2*nx+2*ny+ne,1) = eye(ne);
P(2*nx+2*ny+1:2*nx+2*ny+ne,2*nx+2*ny+1:2*nx+2*ny+ne,2) = eye(ne);
PAI00 = sss; % initial probability distributions of regimes (no regime change, PAI100=1)




% -- Run the Filter -- %
try
    [LLF,~,retcode,Filters] ...
        = ukf_endo_ms_2nd_kim_nelson_create(GLOBALS,xss,yss,Pss,a,P,PAI00,ggammas,data_y,DD,ED,HD,HD2,GD,GD2,V,kf_filtering_level);

    if retcode == 0
        LLF  = -Inf;
        Filters     = 0;
    end
catch
    LLF = -Inf;
    retcode = 0;
    Filters = 0;
end                                         

% -- Set Outputs -- %
LLF_or_Posterior = LLF + log(prior);

if negative == 1
    LLF_or_Posterior = -LLF_or_Posterior;
end

end